# Databricks notebook source
# MAGIC %sh pwd
# COMMAND --------

# Import the configuration
from config.main import config

import requests
import json
import datetime

# COMMAND --------

# TODO: Is todo-to-list working?


def build_exchange_rate_url():
    return "https://data.norges-bank.no/api/data/EXR/B.USD.NOK.SP"


def fetch_exchange_rates(
    start_period: str = "2024-07-09",
    end_period: str = "2025-07-09",
    url: str = build_exchange_rate_url(),
) -> json:
    params = {
        "format": "sdmx-json",
        "startPeriod": start_period,
        "endPeriod": end_period,
        "locale": "no",
    }
    headers = {"Accept": "application/json"}

    response = requests.get(url, params=params, headers=headers)
    return response.json()


def write_data_to_volume(
    data, catalog_name, schema_name, volume_name, filename=None, dry_run=False
):
    if filename is None:
        filename = datetime.datetime.now().strftime("%Y-%m-%d_%H-%M-%S")

    _volume_path = (
        f"/Volumes/{catalog_name}/{schema_name}/{volume_name}/{filename}.json"
    )
    if dry_run:
        print(f"Dry run: would write to {_volume_path}")
        return
    with open(_volume_path, "w") as f:
        json.dump(data, f)


# COMMAND --------

from databricks.connect import DatabricksSession
spark = DatabricksSession.builder.serverless(True).getOrCreate()

# Use configuration from config module
catalog_name = config.get_catalog_name()
schema_name = config.get_schema_name() + "_raw"
team_name = config.get_team_name()
volume_name = "usd_nok_exchange_rate"

print(f"Using catalog: {catalog_name}")
print(f"Using schema: {schema_name}")

spark.sql(f"USE CATALOG {catalog_name}")
spark.sql(f"CREATE SCHEMA IF NOT EXISTS {schema_name}")
spark.sql(f"GRANT MANAGE ON SCHEMA {schema_name} TO {team_name}_developers")
spark.sql(f"GRANT MANAGE ON SCHEMA {schema_name} TO {team_name}_owners")
spark.sql(f"CREATE VOLUME IF NOT EXISTS {schema_name}.{volume_name}")
# COMMAND --------