from databricks.sdk import WorkspaceClient
import os
import getpass
import re
import unicodedata


class Config:
    def __init__(self):
        self.catalog = None
        self.schema = None
        self.team = None
        self.is_in_databricks = False
        self.bundle_target = "dev"

    def get_full_table_name(self, table_name):
        """Return fully qualified table name: catalog.schema.table"""
        return f"{self.catalog}.{self.schema}.{table_name}"

    def get_schema_name(self):
        """Return the schema name"""
        return self.schema

    def get_catalog_name(self):
        """Return the catalog name"""
        return self.catalog

    def get_team_name(self):
        """Return the team name"""
        return self.team
    
    def is_in_databricks_env(self):
        """Check if running in Databricks environment"""
        return self.is_in_databricks
    
    def get_bundle_target(self):
        """Return the bundle target (dev, prod, etc.)"""
        return self.bundle_target

    def __str__(self):
        return f"Config(catalog='{self.catalog}', schema='{self.schema}')"

    def as_dict(self):
        """Return all set configuration properties as a dictionary."""
        return dict(self.__dict__)

def uc_safe_name(name: str) -> str:
    """Convert a name to a safe format for use in Databricks."""
    # return name.replace("-", "_").replace(" ", "_").lower()
    nfkd_form = unicodedata.normalize("NFKD", name)
    only_ascii = nfkd_form.encode("ASCII", "ignore").decode("ASCII")
    cleaned = re.sub(r"[^a-zA-Z0-9_]", "_", only_ascii)
    return cleaned.lower()


# Try to load environment variables from .env file if running locally
if "DATABRICKS_RUNTIME_VERSION" not in os.environ:
    try:
        from dotenv import load_dotenv

        load_dotenv()
        print("Loaded environment variables from .env file")
    except ImportError:
        print("python-dotenv not installed, using system environment variables")
    except Exception as e:
        print(f"Could not load .env file: {e}")

# Initialize the Databricks workspace client only if in Databricks environment
try:
    # Check if we're running in Databricks
    if "DATABRICKS_RUNTIME_VERSION" in os.environ:
        w = WorkspaceClient()
        dbutils = w.dbutils
        is_in_databricks = True
        print("Running in Databricks environment")
    else:
        # Running outside Databricks (local development)
        w = None
        dbutils = None
        is_in_databricks = False
        print("Running in local environment")
except Exception as e:
    print(f"Failed to initialize Databricks client: {e}")
    w = None
    dbutils = None
    is_in_databricks = False


# Configuration setup
if is_in_databricks and dbutils:
    # Running in Databricks - use widgets
    dbutils.widgets.text("bundle.target", "dev", "Bundle Target")
    dbutils.widgets.text("team_name", "team2", "Team Name")
    dbutils.widgets.text("catalog", "", "Catalog")  # Default to empty string
    dbutils.widgets.text("schema", "norges_bank_valutakurser", "Schema")
    dbutils.widgets.text("git_branch", "", "Git Branch")

    bundle_target = dbutils.widgets.get("bundle.target")
    team_name = dbutils.widgets.get("team_name")
    catalog = dbutils.widgets.get("catalog")
    schema_base = dbutils.widgets.get("schema")
    git_branch = dbutils.widgets.get("git_branch")
    # Get current username from Databricks
    try:
        current_user = w.current_user.me().user_name.split("@")[0]
        print(f"Databricks user: {current_user}")
    except Exception as e:
        print(f"Could not get Databricks user, using fallback: {e}")
        current_user = getpass.getuser()
else:
    # Running locally - use environment variables or defaults
    bundle_target = os.environ.get("BUNDLE_TARGET", "dev")
    team_name = os.environ.get("TEAM_NAME", "team2")
    catalog = os.environ.get("CATALOG", "")
    schema_base = os.environ.get("SCHEMA", "main")
    git_branch = os.environ.get("GIT_BRANCH", "")

    # Get current username from system
    current_user = getpass.getuser()
    print(f"Local user: {current_user}")

# Initialize configuration
config = Config()
# If catalog is not set, default to team_name
if not catalog:
    catalog = team_name
config.catalog = catalog

config.team = team_name

if bundle_target == "dev":
    # For development: prefix schema with git branch, username and team
    schema = (
        f"{git_branch}_{current_user}_{schema_base}"
        if git_branch and is_in_databricks
        else f"{current_user}_{schema_base}"
    )
elif bundle_target == "prod":
    # For production: use team and schema base only
    schema = f"{schema_base}"
else:
    # For other environments (staging, etc.)
    schema = f"{bundle_target}_{schema_base}"

config.schema = uc_safe_name(schema)
config.is_in_databricks = is_in_databricks
config.bundle_target = bundle_target

# Print the configuration for debugging
print(f"Bundle Configuration: {config}")
print(f"Full schema path: {config.catalog}.{config.schema}")
print(f"Team: {config.team}")
print(f"Git branch: {git_branch}")

# Make config available globally
globals()["config"] = config
